using System;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents VistA security codes (access/verify). 
	/// It implements checking for disallowed symbols but does not
	/// currently check codes for complexity other than requiring 
	/// 6 characters minimum length. 
	/// </summary>
	public class VistASecurityCode
	{
		private string _code;
		private static readonly char[] DisallowedChars = { ';', '^', ':' };
		private const int MinCodeLength = 6;

		/// <summary>
		/// The only instance constructor creating an instance of security code
		/// from the supplied string. It will throw an exception if string does 
		/// not satisfy to security code requirements. 
		/// General requirements are: security code cannot contain characters ';', '^' and ':' 
		/// and must be at least 6 characters long. 
		/// Note, that this class instances are immutable - once created they 
		/// cannot be modified. 
		/// </summary>
		/// <param name="codeString">Security code represented as string.</param>
		public VistASecurityCode( string codeString )
		{
			if( codeString == null ) 
				throw( new ArgumentNullException( "codeString" ) );

			string _errorMessage;

			if( !IsValidCodeString( codeString, out _errorMessage ) )
				throw( new ArgumentOutOfRangeException( "codeString", SR.Exceptions.VistASecurityCodeSuppliedSecurityStringWasInvalid( _errorMessage ) ) );

			_code = codeString;
		}

		/// <summary>
		/// Validation method. Verifies that input string satisfies to 
		/// to all requirements needed to represent security code. 
		/// </summary>
		/// <param name="sourceString">Input string to validate.</param>
		/// <param name="errorMessage">Output validation error message.</param>
		/// <returns>True if string may be converted to security code. Otherwise - false.</returns>
		public static bool IsValidCodeString( string sourceString, out string errorMessage )
		{
			if( sourceString == null )
			{
				errorMessage = SR.Exceptions.VistASecurityCodeCannotBeConstructedFromNullObject();
				return false;
			}

			if( sourceString.Length < MinCodeLength ) 
			{
				errorMessage = SR.Exceptions.VistASecurityCodeInputStringIsTooShort( sourceString.Length, MinCodeLength );
				return false;
			}

			int _disallowedCharPos;

			if( (_disallowedCharPos = sourceString.IndexOfAny( DisallowedChars )) >= 0 )
			{				
				errorMessage = SR.Exceptions.VistASecurityCodeDisallowedCharFoundInInputString( sourceString[_disallowedCharPos], _disallowedCharPos + 1 );
				return false;
			}

			errorMessage = null;
			return true;
		}

		/// <summary>
		/// Static security code constructor parsing string obfuscated with VistA hash
		/// algorithm and returning new security code instance.
		/// Will throw an exception if input string does not 
		/// satisfy to VistA hash or security code requirements. 
		/// </summary>
		/// <param name="sourceObfuscatedString">Security code string obfuscated with VistA hash algorithm.</param>
		/// <returns>New security code instance.</returns>
		public static VistASecurityCode ParseVistAObfuscatedString( string sourceObfuscatedString )
		{
			if( sourceObfuscatedString == null ) 
				throw( new ArgumentNullException( "sourceObfuscatedString" ) );
			try
			{
				return VistASecurityCode.Parse( VistAObfuscator.Deobfuscate( sourceObfuscatedString ) );
			}
			catch( ArgumentOutOfRangeException xcp )
			{
				throw( new StringParseException( xcp.Message ) );
			}
		}

		/// <summary>
		/// Standard method returning string representation of security code. 
		/// </summary>
		/// <returns>String containing security code.</returns>
		public override string ToString()
		{
			return _code;
		}

		/// <summary>
		/// Converts security code string to its obfuscated representation 
		/// with VistA hash obfuscation algorithm.
		/// </summary>
		/// <returns>Obfuscated security code.</returns>
		public string ToVistAObfuscatedString()
		{
			return VistAObfuscator.Obfuscate( _code );
		}

		/// <summary>
		/// Static factory method. Provides wrapper around the constructor. 
		/// The only difference is that this method throws StringParseException 
		/// instead of ArgumentOutOfRangeException.
		/// </summary>
		/// <param name="codeString">Security code represented as string.</param>
		/// <returns>Newly created instance of VistASecurityCode.</returns>
		public static VistASecurityCode Parse( string codeString )
		{
			if( codeString == null )
				throw( new ArgumentNullException( "codeString" ) );

			try
			{
				return new VistASecurityCode( codeString );
			}
			catch( ArgumentOutOfRangeException xcp )
			{
				throw( new StringParseException( xcp.Message ) );
			}
		}
	}
}
